
/************************************************************************
 *																								*
 *		Z80 Disassembler - Copyright (C) 1996 by									*
 *		Jeffery L. Post																	*
 *		22726 Benner Ave.																	*
 *		Torrance, CA  90505																*
 *																								*
 *		DZ80P1.C - Disassembly Passes 0 & 1											*
 *																								*
 *		Version 2.0 - 08/19/96															*
 *																								*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.												*
 *																								*
 *	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.									*
 *																								*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the Free Software				*
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
 *																								*
 ************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	"dz80.h"

#ifdef	MSDOS
#include	<dos.h>
#include	<alloc.h>
#endif

#ifdef	LINUX
#include	<malloc.h>
#endif

#define	WIDGET_TIME	500

/*********************
 *							*
 *		Prototypes		*
 *							*
 *********************/

void	pass0(void);
void	pass1(void);
char	*get_adrs(char *text, word *val);
void	dump_ascii(word adrs);
void	dump_bytes(word adrs);
struct sym	*sort(struct sym *list, SYM_PTR *array, word count);
struct sym	*sort_by_name(struct sym *list);
struct sym	*sort_by_value(struct sym *list);
struct sym	*merge_by_name(struct sym *a, struct sym *b);
struct sym	*merge_by_value(struct sym *a, struct sym *b);
void	chk_dup_name(struct sym *list, word count);
void	chk_dup_value(struct sym *list, word count);
void	do_widget(void);

extern void	puthex(word j);
extern int	ascii(int i);
extern char	*find_entry(word val, word count, SYM_PTR *table);
extern void	add_entry(word val, char *symbol, int type);

/***************************
 *									*
 *		Global variables		*
 *									*
 ***************************/

char	widget;
word	widget_count;

SYM_PTR	*sym_val_index;				/* array of pointers				*/
SYM_PTR	*lab_val_index;				/*  for binary search			*/
struct sym	*tail_ptr, *head_ptr;	/* sort pointers					*/

extern FILE	*fp;							/* dos file struct				*/
extern char	ctl[32];						/* control file name				*/
extern int	hexflag;						/* append hex flag				*/
extern int	kcnt;							/* output char counter			*/
extern word	himark;						/* highest valid adrs			*/
extern word	offset;						/* program counter offset		*/
extern byte	*pmem;						/* program data pointer			*/
extern byte	*pflg;						/* pointer to program flags	*/
extern char	defbstr[8];					/* string for defined bytes	*/
extern char	defwstr[8];					/* string for defined words	*/
extern char	ascistr[8];					/* string for defined ascii	*/
extern char	string[ASCLIMIT];			/* ascii output buffer			*/
extern word	asc_cnt;						/* count for string data		*/
extern byte	byte_data[BYTELIMIT];	/* binary data for defb			*/
extern word	byte_cnt;					/* count for binary data		*/
extern word	word_data[WORDLIMIT];	/* binary data for defw			*/
extern word	word_cnt;					/* count for word data			*/
extern byte	dump;							/* dump just done flag			*/
extern byte opttbl[256];				/* code option bit flags		*/
extern char	linebuffer[128];			/* input line buffer				*/
extern struct sym	*sym_tab;			/* symbol table pointer			*/
extern struct sym	*lab_tab;			/* label table pointer			*/
extern struct sym	*sym_tab_last;		/* last symbol table pointer	*/
extern struct sym	*lab_tab_last;		/* last label table pointer	*/
extern word	symbol_count;				/* number of symbols				*/
extern word	label_count;				/* number of labels				*/
extern char	linebuffer[128];			/* input line buffer				*/

extern unsigned char edcode[256];
extern unsigned char ddcode[256];

/*********************
 *							*
 *			Code			*
 *							*
 *********************/

/************************************************************************
 *																								*
 *		Read control file, if it exists, and flag areas as code, text,		*
 *		or whatever. Also handle sfr name changes, labels, etc.				*
 *																								*
 *		Some strange things happen here with oring and anding on the		*
 *		flag bits; this is so the user can define something as code,		*
 *		data, or whatever, and assign a label for the same location.		*
 *		We should handle the result intelligently regardless of the			*
 *		order in which such things are found in the control file.			*
 *		Defining a location as code sets the PF_NOINIT bit, which			*
 *		might seem strange at first, but this is to prevent pass 2 from	*
 *		skipping over multiple NOPs if the user specifically defines		*
 *		them as code.																		*
 *																								*
 ************************************************************************/

void pass0(void)
{
	char	*text, func, c, *ltext;
	word	start, stop, code, temp;

	widget_count = 0;
	widget = '/';
	fp = fopen(ctl, "r");
	if (fp != NULL)			/* if control file exists... */
	{
		printf("\nReading control file   ");
		while (!feof(fp))						/* until end of file... */
		{
			start = stop = 0;
			*linebuffer = '\0';				/* clear previous line */
			fgets(linebuffer, 127, fp);	/* read one line */
			text = &linebuffer[1];
			text = get_adrs(text, &start);
			while (1)
			{
				c = *text++;
				if (c != ' ' && c != '\t')	/* skip whitespace */
					break;
			}
			if (c == '\n' || c == ';')		/* if only one numeric... */
				--text;							/* back up to newline */
			func = c;							/* save operator */
			ltext = text;
			--ltext;
			text = get_adrs(text, &stop);
			if (func == '+')					/* check for valid operator */
				stop += (start - 1);
			else if (func == '-' && !stop)
				stop = start;

			switch (toupper(linebuffer[0]))
			{
				case 'A':								/* address */
					do
					{						/* get address to reference */
						code = ((word) pmem[start + 1]) << 8;
						temp = pmem[start] & 0xff;
						code |= temp;
						pflg[code] |= PF_REF;		/* flag referenced address */
						pflg[code] &= ~(PF_CLREF | PF_SPLIT);
						pflg[start] |= PF_ADRS;		/* set flags to adrs code */
						pflg[start++] &=
							~(PF_NOINIT | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
						pflg[start] |= PF_ADRS;
						pflg[start++] &=
							~(PF_NOINIT | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
					} while (start < stop);
					break;

				case 'B':								/* byte binary */
					do
					{
						pflg[start] |= PF_BYTE;
						pflg[start++] &=
							~(PF_NOINIT | PF_ADRS | PF_WORD | PF_ASCII | PF_SPLIT);
					} while (start <= stop);
					break;

				case 'C':								/* code */
					do
					{
						pflg[start] |= PF_NOINIT;	/* to catch 00 or FF bytes */
						pflg[start++] &=
							~(PF_ADRS | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
					} while (start <= stop);
					break;

				case 'I':								/* ignore initialized data */
					do
					{
						pflg[start++] = PF_INIT;
					} while (start <= stop);
					break;

				case 'L':								/* label */
					pflg[start] |= PF_REF;			/* flag as referenced */
					pflg[start] &= ~(PF_CLREF | PF_SPLIT);	/* not split (yet) */
					if (isalnum(*ltext))
						add_entry(start, ltext, LABEL);
					break;

				case 'S':								/* symbol */
					add_entry(start, ltext, SYMBOL);
					break;

				case 'T':								/* text */
					do
					{
						pflg[start] |= PF_ASCII;
						pflg[start++] &=
							~(PF_NOINIT | PF_ADRS | PF_WORD | PF_BYTE | PF_SPLIT);
					} while (start <= stop);
					break;

				case 'W':								/* word binary */
					do
					{
						pflg[start] |= PF_WORD;
						pflg[start++] &=
							~(PF_NOINIT | PF_ADRS | PF_BYTE | PF_ASCII | PF_SPLIT);
					} while (start <= stop);
					break;

				case 0x00:								/* ignore empty lines */
				case '\n':
				case '\r':
				case '\t':
				case ' ':
				case ';':								/* ignore commented out lines */
					break;

				default:						/* somebody didn't read the docs... */
					printf("\rUnknown control code: 0x%02x in '%s'\n",
						linebuffer[0], linebuffer);
					break;
			}
			widget_count++;
			if (widget_count > WIDGET_TIME)
			{
				widget_count = 0;
				do_widget();
			}
		}
		widget_count = 0;
		if (label_count || symbol_count)	/* set up tail node for sort */
		{
			tail_ptr = (struct sym *) malloc(sizeof(struct sym) + 12);
			tail_ptr->next = tail_ptr;
			tail_ptr->name[0] = 0xfe;		/* set max values for sort */
			tail_ptr->name[1] = 0;
			tail_ptr->val = 0xffff;
		}
		if (label_count)					/* if labels encountered... */
		{
			printf("\r                        ");
			printf("\rSorting labels   ");
			lab_tab_last->next = tail_ptr;	/* set up pointer array for sort */
			lab_val_index = malloc(sizeof(SYM_PTR) * label_count);
			if (lab_val_index == NULL)
			{
				printf("\nINTERNAL ERROR! - No memory for label pointers");
				exit(MEM_ERROR);
			}
			lab_tab = sort(lab_tab, lab_val_index, label_count);
		}
		if (symbol_count)					/* if symbols encountered... */
		{
			printf("\r                        ");
			printf("\rSorting symbols   ");
			sym_tab_last->next = tail_ptr;
			sym_val_index = malloc(sizeof(SYM_PTR) * symbol_count);
			if (sym_val_index == NULL)
			{
				printf("\nINTERNAL ERROR! - No memory for symbol pointers");
				exit(MEM_ERROR);
			}
			sym_tab = sort(sym_tab, sym_val_index, symbol_count);
		}
	}
}

void do_widget(void)
{
	printf("\b%c", widget);
	switch (widget)
	{
		case '/':
			widget = '|';
			break;

		case '|':
			widget = '\\';
			break;

		case '\\':
			widget = '-';
			break;

		default:
			widget = '/';
			break;
	}
}

/*********************************************************
 *																			*
 *		Get hexadecimal number from line in control file.	*
 *		Return updated character pointer.						*
 *																			*
 *********************************************************/

char *get_adrs(char *text, word *val)
{
	word	result, start;
	char	c;

	result = start = 0;
	c = toupper(*text);
	while (c)
	{
		if (c == ';')			/* beginning of comment, ignore all else */
			break;
		if (c == '\n')			/* necessary because isspace() includes \n */
			break;
		if (isspace(c))		/* skip leading whitespace */
		{
			text++;
			if (start)			/* if result already begun... */
				break;
		}
		else if (!isxdigit(c))		/* done if not hexadecimal character */
			break;
		else
		{
			start = 1;			/* flag beginning of result conversion */
			c = (c > '9') ? c - 0x37 : c - 0x30;
			result <<= 4;
			result |= ((word) c & 0xf);
			text++;
		}
		c = toupper(*text);
	}
	*val = result;				/* pass number back to caller */
	return(text);				/* and return updated text pointer */
}

/************************************************************
 *																				*
 *					Sort label or symbol table							*
 *																				*
 *	First sort by name so that we can check for duplicates,	*
 *	then sort by value, check for duplicates, and set up		*
 *	pointer array for binary search.									*
 *																				*
 ************************************************************/

struct sym *sort(struct sym *list, SYM_PTR *array, word count)
{
	word	i;
	struct sym	*sptr, *temp;

	sptr = sort_by_name(list);
	chk_dup_name(sptr, count);
	sptr = sort_by_value(sptr);
	chk_dup_value(sptr, count);
	temp = sptr;
	for (i=0; i<count; i++)		/* set up array of pointers sorted by value */
	{
		array[i] = temp;
		temp = temp->next;
	}
	return(sptr);
}

/******************************************************************
 *																						*
 *		In-place non-recursive merge sort using label text as key	*
 *																						*
 ******************************************************************/

struct sym *sort_by_name(struct sym *list)
{
	word			i, n;
	struct sym	*a, *b, *todo, *t;

	head_ptr = (struct sym *) malloc(sizeof(struct sym) + 12);
	head_ptr->next = list;
	a = tail_ptr;
	for (n=1; a != head_ptr->next; n = n + n)
	{
		todo = head_ptr->next;
		list = head_ptr;
		while (todo != tail_ptr)
		{
			t = todo;
			a = t;
			for (i=1; i<n; i++)
				t = t->next;
			b = t->next;
			t->next = tail_ptr;
			t = b;
			for (i=1; i<n; i++)
				t = t->next;
			todo = t->next;
			t->next = tail_ptr;
			list->next = merge_by_name(a, b);
			for (i=1; i<=n+n; i++)
				list = list->next;
		}
	}
	return(head_ptr->next);
}

/***************************************************************
 *																					*
 *		In-place non-recursive merge sort using value as key		*
 *																					*
 ***************************************************************/

struct sym *sort_by_value(struct sym *list)
{
	word			i, n;
	struct sym	*a, *b, *todo, *t;

	head_ptr = (struct sym *) malloc(sizeof(struct sym));
	head_ptr->next = list;
	a = tail_ptr;
	for (n=1; a != head_ptr->next; n = n + n)
	{
		todo = head_ptr->next;
		list = head_ptr;
		while (todo != tail_ptr)
		{
			t = todo;
			a = t;
			for (i=1; i<n; i++)
				t = t->next;
			b = t->next;
			t->next = tail_ptr;
			t = b;
			for (i=1; i<n; i++)
				t = t->next;
			todo = t->next;
			t->next = tail_ptr;
			list->next = merge_by_value(a, b);
			for (i=1; i<=n+n; i++)
				list = list->next;
		}
	}
	return(head_ptr->next);
}

/***************************************
 *													*
 *		Merge sub-lists by text field		*
 *													*
 ***************************************/

struct sym *merge_by_name(struct sym *a, struct sym *b)
{
	int			i;
	struct sym	*c;

	widget_count++;
	if (widget_count > WIDGET_TIME)
	{
		widget_count = 0;
		do_widget();
	}

	c = tail_ptr;
	do
	{

#ifdef	LINUX
		i = strcasecmp(a->name, b->name);
#else
		i = stricmp(a->name, b->name);
#endif
		if (i <= 0)
		{
			c->next = a;
			c = a;
			a = a->next;
		}
		else
		{
			c->next = b;
			c = b;
			b = b->next;
		}
	} while (c != tail_ptr);
	c = tail_ptr->next;
	tail_ptr->next = tail_ptr;
	return(c);
}

/******************************************
 *														*
 *		Merge sub-lists by value field		*
 *														*
 ******************************************/

struct sym *merge_by_value(struct sym *a, struct sym *b)
{
	struct sym	*c;

	widget_count++;
	if (widget_count > WIDGET_TIME)
	{
		widget_count = 0;
		do_widget();
	}

	c = tail_ptr;
	do
	{
		if (a->val < b->val)
		{
			c->next = a;
			c = a;
			a = a->next;
		}
		else
		{
			c->next = b;
			c = b;
			b = b->next;
		}
	} while (c != tail_ptr);
	c = tail_ptr->next;
	tail_ptr->next = tail_ptr;
	return(c);
}

/******************************************************
 *																		*
 *		Check for redefinitions of label/symbol names	*
 *																		*
 ******************************************************/

void chk_dup_name(struct sym *list, word count)
{
	word	i;

	for (i=0; i<count; i++)
	{

#ifdef	LINUX
		if (!strcasecmp(list->name, list->next->name))
#else
		if (!stricmp(list->name, list->next->name))
#endif
		{
			printf("\nAttempted redefinition of '%s', value 0x%x,"
					 " as value 0x%x\n",
					list->name, list->val, list->next->val);
			exit(USER_ERROR);
		}
		list = list->next;
	}
}

/*********************************************
 *															*
 *		Check for redefinitions of values		*
 *															*
 *********************************************/

void chk_dup_value(struct sym *list, word count)
{
	word	i;

	for (i=0; i<count; i++)
	{
		if (list->val == list->next->val)
		{
			printf("\nAttempted redefinition of value 0x%x, '%s', as '%s'\n",
					list->val, list->name, list->next->name);
			exit(USER_ERROR);
		}
		list = list->next;
	}
}

/***********************************************************************

	Pass one of disassembly. Examine opcodes for internal references
	to other memory locations. If such references are found, flag the
	referenced location so that a label can be generated in the output
	file during pass two.

************************************************************************/

void pass1(void)
{
	word	i, l, pc, rel;
	byte	j, k, mask;

	printf("Pass 1 0000");
	for (i=offset; i<himark; )
	{
		l = i & 0xff;
		k = pmem[i];						/* get stored opcode */

		if (pflg[i] == PF_INIT)			/* ignore un-initialized data */
			i++;
		else if (!(pflg[i] & (PF_ADRS | PF_WORD | PF_BYTE | PF_ASCII)))
		{										/* if code... */
			mask = (byte) (PF_CLREF | PF_SPLIT);
			j = opttbl[k];					/* get option byte */

			if (j & OPT_DIR)				/* if absolute memory reference */
			{
				pc = (pmem[i+1] & 0xff) | ((pmem[i+2] << 8) & 0xff00);
				if (pc >= offset && pc <= himark)
					mask |= PF_NOINIT;
				pflg[pc] = (pflg[pc] & ~mask) | PF_REF;	/* flag reference */
			}

			if (j & OPT_REL)			/* if relative memory reference */
			{
				j = pmem[i+1];			/* get offset */
				rel = (j > 0x7f) ? j | 0xff00 : j & 0xff;	/* sign extend offset */
				pc = i + 2;
				pc += rel;
				if (pc >= offset && pc <= himark)
					mask |= PF_NOINIT;
				pflg[pc] = (pflg[pc] & ~mask) | PF_REF;	/* flag reference */
			}

			if (j & OPT_SPEC)			/* if special processing */
			{
				j = pmem[i+1];			/* get second byte */
				switch (k)
				{
					case 0xed:
						if (edcode[j] & 2)
						{
							pc = (pmem[i+2] & 0xff) | ((pmem[i+3] << 8) & 0xff00);
							if (pc >= offset && pc <= himark)
								mask |= PF_NOINIT;
							pflg[pc] = (pflg[pc] & ~mask) | PF_REF;
						}
						i += (edcode[j] & 3);
						break;

					case 0xdd:
					case 0xfd:
						if (j == 0x21 || j == 0x22 || j == 0x2a)
						{
							pc = (pmem[i+2] & 0xff) | ((pmem[i+3] << 8) & 0xff00);
							if (pc >= offset && pc <= himark)
								mask |= PF_NOINIT;
							pflg[pc] = (pflg[pc] & ~PF_CLREF) | PF_REF;
						}
						i += (ddcode[j] & 3);
						break;
				}
			}
			i = i + (opttbl[k] & OPT_SIZE) + 1;		/* update location pointer */
		}
		else
			i++;
		if ((i & 0xff) < l)
			printf("\rPass 1 %04x", i & 0xff00);
	}
	printf("\rPass 1 - Reference search complete");
}														/*  End of Pass 1 */

/************************************************
 *																*
 *		Output ascii data accumulated in buffer	*
 *																*
 ************************************************/

void dump_ascii(word adrs)
{
	word	padrs, off, cnt;
	char	*cptr;

	padrs = adrs - asc_cnt;			/* print address for comment field */
	adrs = padrs;						/* address in program array */
	cnt = off = 0;						/* cnt = char count, off = buffer offset */
	while (asc_cnt)					/* while data in ascii buffer... */
	{
		if (pflg[adrs] & PF_REF)		/* if addresss is referenced... */
		{
			if (cnt)
			{
				putc('\'', fp);			/* terminate line */
				kcnt++;
				if (hexflag)				/* if comment field requested... */
				{
					do								/* show hex address */
					{
						putc('\t', fp);		/* but tab out to field first */
						kcnt = (kcnt + 8) & 0x78;
					} while (kcnt < XSTOP);

					fprintf(fp, "; %04x", padrs);
				}
				padrs += cnt;					/* update print address */
				cnt = 0;							/* clear char count for this line */
			}
			cptr = find_entry(adrs, label_count, lab_val_index);
												/* see if label exists for this adrs */
			if (cptr == NULL)				/* if not, show address in hex */
				fprintf(fp, "\nX%04x:\t%s\t'", adrs, ascistr);
			else								/* else show label name */
				fprintf(fp, "\n%s:\t%s\t'", cptr, ascistr);
			kcnt = 17;
		}
		else if (!cnt)
		{
			fprintf(fp, "\n\t%s\t'", ascistr);
			kcnt = 17;
		}
		putc(string[off], fp);			/* output data in ascii */
		kcnt++;								/* character position in this line */
		cnt++;								/* increment char in this line */
		off++;								/* increment offset into asci buffer */
		adrs++;								/* offset into program memory */
		if (cnt >= ASCLINE)				/* if max characters per line... */
		{
			putc('\'', fp);				/* terminate line */
			kcnt++;
			if (hexflag)					/* if comment field requested */
			{
				do									/* show hex address */
				{
					putc('\t', fp);
					kcnt = (kcnt + 8) & 0x78;
				} while (kcnt < XSTOP);

				fprintf(fp, "; %04x", padrs);
			}
			padrs += cnt;					/* update print address */
			cnt = 0;
		}
		--asc_cnt;
	}
	putc('\'', fp);					/* terminate line */
	kcnt++;
	if (hexflag && cnt)				/* if comment field requested... */
	{
		do										/* show address */
		{
			putc('\t', fp);
			kcnt = (kcnt + 8) & 0x78;
		} while (kcnt < XSTOP);

		fprintf(fp, "; %04x", padrs);
	}
	dump = 1;
}

/************************************************
 *																*
 *		Output binary data accumulated in buffer	*
 *																*
 ************************************************/

void dump_bytes(word adrs)
{
	word	padrs, bcnt, off, k;
	char	*cptr, chr;

	padrs = adrs - byte_cnt;			/* compute adrs to print in ascii part */
	adrs = padrs;
	bcnt = off = 0;						/* no bytes output yet */
	while (byte_cnt)						/* while data in binary buffer... */
	{
		if (pflg[adrs] & PF_REF)		/* if data adrs is referenced... */
		{
			if (off && hexflag)				/* dump any remaining ascii first */
			{
				do
				{
					putc('\t', fp);
					kcnt = (kcnt + 8) & 0x78;
				} while (kcnt < XSTOP);

				fprintf(fp, "; %04x ", padrs);
				for (k=0; k<off; k++)
					putc(ascii(pmem[padrs + k]), fp);
				padrs += k;						/* update print address */
				off = 0;
			}
			cptr = find_entry(adrs, label_count, lab_val_index);	/* then do a label */
			if (cptr == NULL)
				fprintf(fp, "\nX%04x:\t%s\t", adrs, defbstr);
			else
				fprintf(fp, "\n%s:\t%s\t", cptr, defbstr);
			kcnt = 16;
			bcnt = 0;
		}
		else if (!bcnt)						/* else if first byte... */
		{
			kcnt = 16;
			fprintf(fp, "\n\t%s\t", defbstr);
		}
		else
		{
			putc(',', fp);						/* else separate bytes */
			kcnt++;
		}
		cptr = find_entry(pmem[adrs], symbol_count, sym_val_index);
		if (cptr)
			kcnt += fprintf(fp, "%s", cptr);
		else
		{
			if (!pflg[adrs] & PF_ASCII)
				puthex(pmem[adrs]);
			else									/* user defined this as ascii text */
			{										/* even though it's not; let's try */
				chr = pmem[adrs];				/* to give him what he wants. */
				if (chr & 0x80)				/* if flagged binary byte because */
				{									/* high bit is set... */
					chr &= 0x7f;
					if (chr >= ' ' && chr <= 'z')		/* would it be ascii */
					{											/* without bit 7 set? */
						kcnt += fprintf(fp, "'%c'+80h", chr);	/* yes */
						bcnt += 3;
					}
					else							/* else do as binary and remove */
					{								/* ascii flag */
						puthex(pmem[adrs]);
						pflg[adrs] &= ~PF_ASCII;
					}
				}
				else								/* high bit not set, so is */
				{									/* really binary, not ascii */
					puthex(pmem[adrs]);
					pflg[adrs] &= ~PF_ASCII;
				}
			}
		}
		bcnt++;
		if (bcnt >= BYTELINE)				/* if max bytes per line... */
		{
			bcnt = 0;
			if (hexflag)						/* do ascii dump of previous bytes */
			{
				do
				{
					putc('\t', fp);
					kcnt = (kcnt + 8) & 0x78;
				} while (kcnt < XSTOP);

				fprintf(fp, "; %04x ", padrs);
				for (k=0; k<BYTELINE; k++)
				{
					if (!(pflg[padrs + k] & PF_ASCII))		/* don't show ascii */
						putc(ascii(pmem[padrs + k]), fp);	/* if already ascii */
				}
				padrs += k;
				off = 0;
			}
		}
		else
			off++;
		--byte_cnt;
		adrs++;
	}
	if (off && hexflag)					/* generate comment line */
	{
		do
		{
			putc('\t', fp);
			kcnt = (kcnt + 8) & 0x78;
		} while (kcnt < XSTOP);

		fprintf(fp, "; %04x ", padrs);	/* show address and ascii for data */
		for (k=0; k<off; k++)
		{
			if (!(pflg[padrs + k] & PF_ASCII))		/* don't show ascii */
				putc(ascii(pmem[padrs + k]), fp);	/* if already ascii */
		}
	}
	dump = 1;
}

/* end of dz80p1.c */

